/*
Copyright 2008-2009 Elöd Egyed-Zsigmond, Cyril Laitang
Copyright 2009-2011 Samuel Gesche

This file is part of IPRI News Analyzer.

IPRI News Analyzer is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

IPRI News Analyzer is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with IPRI News Analyzer.  If not, see <http://www.gnu.org/licenses/>.
*/

package data.base.connectors;

import java.sql.ResultSet;
import java.sql.SQLException;

import java.util.ArrayList;
import java.util.Vector;
import java.util.Set;
import java.util.HashSet;
import java.util.Map;
import java.util.HashMap;

import data.base.Database;
import data.base.NoBaseException;
import data.structures.tagging.LemmaItem;
import proc.text.Horloge;

public class LocalSemanticDatabase {

    private Database myDB;

    public LocalSemanticDatabase(Database db) {
        myDB = db;
    }

    private static Map<String, Set<LemmaItem>> goDejaTrouvees = new HashMap<String, Set<LemmaItem>>();
    /**
     * Retourne sous forme lemmatisée la catégorie de l'ontologie correspondant à
     * la généralisation au niveau donné du lemme donné.
     * @param lemme un lemme à généraliser
     * @return sous forme lemmatisée la catégorie de l'ontologie correspondant à
     * la généralisation au niveau donné du lemme donné.
     */
    public Set<LemmaItem> getGeneralisationOnto(LemmaItem lemme) throws NoBaseException {
        System.out.println(Horloge.getHMS() + "     Généralisation");
        Set<LemmaItem> res = goDejaTrouvees.get(lemme.getLemmaName());
        if (res == null) {
            res = new HashSet<LemmaItem>();
            System.out.println(Horloge.getHMS() + "     Recherche des concepts");
            String s = getBestConcept(lemme);
            res.addAll(getLemmesDeLaCategorieOnto(getCategorieOnto(s)));
            goDejaTrouvees.put(lemme.getLemmaName(), res);
            System.out.println(Horloge.getHMS() + "     Généralisation : "+lemme.getLemmaName()+" - "+goDejaTrouvees.get(lemme.getLemmaName()));
        } else {
            System.out.println(Horloge.getHMS() + "     Récupération des résultats précédents");
        }
        return res;
    }

    private static Map<String, Set<LemmaItem>> gwDejaTrouvees = new HashMap<String, Set<LemmaItem>>();
    /**
     * Retourne sous forme lemmatisée la(les) catégorie(s) wiki correspondant à
     * la généralisation du lemme donné.
     * @param lemme un lemme à catégoriser
     * @return sous forme lemmatisée la(les) catégorie(s) wiki correspondant à
     * la généralisation du lemme donné.
     */
    public Set<LemmaItem> getGeneralisationWiki(LemmaItem lemme) throws NoBaseException {
            System.out.println(Horloge.getHMS() + "     Catégorisation");
        Set<LemmaItem> res = gwDejaTrouvees.get(lemme.getLemmaName());
        if (res == null) {
            res = new HashSet<LemmaItem>();
            System.out.println(Horloge.getHMS() + "     Recherche des concepts");
            String s = getBestConcept(lemme);
            String[] s2 = getCategoriesWiki(s);
            for (int j = 0; j < s2.length; j++) {
                res.addAll(getLemmesDeLaCategorieWiki(s2[j]));
            }
            gwDejaTrouvees.put(lemme.getLemmaName(), res);
            System.out.println(Horloge.getHMS() + "     Généralisation : "+lemme.getLemmaName()+" - "+gwDejaTrouvees.get(lemme.getLemmaName()));
        } else {
            System.out.println(Horloge.getHMS() + "     Récupération des résultats précédents");
        }
        return res;
    }

    private static Map<String, Set<LemmaItem>> csDejaTrouves = new HashMap<String, Set<LemmaItem>>();
    /**
     * Retourne sous forme lemmatisée le champ sémantique du lemme donné.
     * @param lemme un lemme à catégoriser
     * @return sous forme lemmatisée le champ sémantique du lemme donné.
     */
    public Set<LemmaItem> getChampSemantique(LemmaItem lemme) throws NoBaseException {
        System.out.println(Horloge.getHMS() + "      Champ sémantique");
        Set<LemmaItem> res = csDejaTrouves.get(lemme.getLemmaName());
        if (res == null) {
            res = new HashSet<LemmaItem>();
            String s = getBestConcept(lemme);
            //for(int i=0; i<s.length; i++){
            String query = "SELECT l.lemme, l.countTitle, l.countDesc, l.Lex" +
                    " FROM ipri_onto_concepts c, " +
                    "ipri_onto_con_lemmes l WHERE c.ConceptID=l.idConcept AND c.ResourceID=2 AND " +
                    "c.ConceptName='" + proc.text.Codecs.escapeHTML(s) + "';";
            //System.out.println(Horloge.getHMS() + "      "+query);
            ResultSet rs = myDB.executeSelection(query);
            Vector<LemmaItem> res1 = new Vector<LemmaItem>();
            try {
                while (rs.next()) {
                    LemmaItem l = new LemmaItem();
                    l.setLemmaName(rs.getString(1).trim());
                    l.setCountTitle(rs.getInt(2));
                    l.setCountDesc(rs.getInt(3));
                    l.setLemmaLex(rs.getString(4).trim());
                    res1.add(l);
                }
            } catch (SQLException e) {
                System.err.println("Erreur de récupération des concepts contenant le lemme " + lemme + ".");
            }
            LemmaItem[] res2 = new LemmaItem[res1.size()];
            res1.toArray(res2);
            for (int j = 0; j < res2.length; j++) {
                res.add(res2[j]);
            }
            //}
            System.out.println(Horloge.getHMS() + "      Champ sémantique trouvé : " + res.size() + " nouveaux lemmes "+res+".");
            csDejaTrouves.put(lemme.getLemmaName(), res);
        } else {
            System.out.println(Horloge.getHMS() + "      Récupération des résultats précédents");
        }
        return res;
    }

    private String[] getConcepts(LemmaItem lemme) throws NoBaseException {
        String query = String.format("SELECT c.ConceptName FROM ipri_onto_concepts c, " +
                "ipri_onto_con_lemmes l WHERE c.ConceptID=l.idConcept AND c.ResourceID=2 AND " +
                "l.lemme='"+proc.text.Codecs.escapeHTML(lemme.getLemmaName())+"' AND " +
                "l.countTitle>0;"/* AND l.Lex='"+lemme.getLemmaLex()+"';"*/);
        System.out.println(Horloge.getHMS()+"      Interrogation de la base");
        ResultSet rs = myDB.executeSelection(query);
        Vector<String> res = new Vector<String>();
        try {
            while (rs.next()) {
                res.add(rs.getString(1).trim());
            }
        } catch (SQLException e) {
            System.err.println("Erreur de récupération des concepts contenant le lemme "+lemme+".");
        }
        String[] res2 = new String[res.size()];
        res.toArray(res2);
        System.out.println(Horloge.getHMS()+"      Interrogation de la base terminée : "+res2.length+" concepts trouvés.");
        return res2;
    }

    private static Map<String, String> bcDejaCalcules = new HashMap();
    /**
     * Récupération des concepts proches (contenant le lemme dans le titre),
     * puis choix du plus proche par distance d'édition.
     * @param lemme le lemme à chercher
     * @return le concept Wikipédia le plus proche
     */
    private String getBestConcept(LemmaItem lemme) throws NoBaseException {
        String res = bcDejaCalcules.get(lemme.getLemmaName());
        if (res == null) {
            String[] s = getConcepts(lemme);
            if (s.length == 0) {
                return "";
            }
            int dist = Integer.MAX_VALUE;
            int index = 0;
            for (int i = 0; i < s.length; i++) {
                int d = distanceEdition(lemme.getLemmaName(), s[i]);
                if (d < dist) {
                    dist = d;
                    index = i;
                }
            }
            System.out.println(Horloge.getHMS()+"      Choix : " + s[index] + ".");
            res = s[index];
            bcDejaCalcules.put(lemme.getLemmaName(), res);
        }
        return res;
    }

    private int minimum(int a, int b, int c) {
        return Math.min(Math.min(a, b), c);
    }

    public int distanceEdition(CharSequence str1,
            CharSequence str2) {
        int[][] distance = new int[str1.length() + 1][str2.length() + 1];

        for (int i = 0; i <= str1.length(); i++) {
            distance[i][0] = i;
        }
        for (int j = 0; j <= str2.length(); j++) {
            distance[0][j] = j;
        }

        for (int i = 1; i <= str1.length(); i++) {
            for (int j = 1; j <= str2.length(); j++) {
                distance[i][j] = minimum(
                        distance[i - 1][j] + 1,
                        distance[i][j - 1] + 1,
                        distance[i - 1][j - 1] + ((str1.charAt(i - 1) == str2.charAt(j - 1)) ? 0
                        : 1));
            }
        }

        return distance[str1.length()][str2.length()];
    }


    private Set<LemmaItem> getLemmesDuConcept(String concept) throws NoBaseException {
        String query = String.format("SELECT l.lemme, l.countTitle, l.countDesc, l.Lex " +
                "FROM ipri_onto_concepts c, ipri_onto_con_lemmes l " +
                "WHERE c.ConceptID=l.idConcept AND c.ResourceID=2 AND " +
                "c.conceptName='"+proc.text.Codecs.escapeHTML(concept)+"';");
        ResultSet rs = myDB.executeSelection(query);
        Set<LemmaItem> res = new HashSet<LemmaItem>();
        try {
            while (rs.next()) {
                LemmaItem l = new LemmaItem();
                l.setLemmaName(rs.getString(1).trim());
                l.setCountTitle(rs.getInt(2));
                l.setCountDesc(rs.getInt(3));
                l.setLemmaLex(rs.getString(4).trim());
                res.add(l);
            }
        } catch (SQLException e) {
            System.err.println("Erreur de récupération des lemmes du concept "+concept+".");
        }
        return res;
    }

    private String getCategorieOnto(int concept) throws NoBaseException {
        String query = "SELECT o.ressource FROM ipri_ontologie o, ipri_onto_con_cate_onto c " +
                "WHERE c.idConcept="+concept+" AND o.id=c.idCateOnto;";
        //System.out.println("      Recherche de la catégorie.");
        ResultSet rs = myDB.executeSelection(query);
        String res = "";
        try {
            if (rs.next()) {
                res = rs.getString(1).trim();
            }
        } catch (SQLException e) {
            System.err.println("Erreur de récupération de la catégorie ontologique du concept "+concept+".");
        }
        //System.out.println("      Catégorie trouvée.");
        return res;
    }

    private String getCategorieOnto(String concept) throws NoBaseException {
        String query = "SELECT c.CategorieName FROM ipri_onto_categories c " +
                "WHERE c.ConceptName='"+proc.text.Codecs.escapeHTML(concept)+"';";
        ResultSet rs = myDB.executeSelection(query);
        String res = "";
        try {
            if (rs.next()) {
                res = rs.getString(1).trim();
            }
        } catch (SQLException e) {
            System.err.println("Erreur de récupération de la catégorie ontologique du concept "+concept+".");
        }
        return res;
    }

    private Set<LemmaItem> getLemmesDeLaCategorieOnto(String categorieOnto) throws NoBaseException {
        //return new HashSet<LemmaItem>();
        String query = String.format("SELECT l.lemme, l.countTitle, l.countDesc, l.Lex " +
                "FROM ipri_onto_lemmes_en l, ipri_ontologie o " +
                "WHERE o.id=l.idConcept AND o.ressource='"+categorieOnto+"';");
        //System.out.println("      Recherche des lemmes.");
        ResultSet rs = myDB.executeSelection(query);
        Set<LemmaItem> res = new HashSet<LemmaItem>();
        try {
            while (rs.next()) {
                LemmaItem l = new LemmaItem();
                l.setLemmaName(rs.getString(1).trim());
                l.setCountTitle(rs.getInt(2));
                l.setCountDesc(rs.getInt(3));
                l.setLemmaLex(rs.getString(4).trim());
                res.add(l);
            }
        } catch (SQLException e) {
            System.err.println("Erreur de récupération des lemmes de la catégorie "+categorieOnto+".");
        }
        //System.out.println("      "+res.size()+" lemmes trouvés.");
        return res;
    }

    private String[] getCategoriesWiki(String concept) throws NoBaseException {
        String query = "SELECT c.CategorieName FROM ipri_onto_categories2 c " +
                "WHERE c.ConceptName='"+proc.text.Codecs.escapeHTML(concept)+"';";
        ResultSet rs = myDB.executeSelection(query);
        Vector<String> res = new Vector<String>();
        try {
            while (rs.next()) {
                res.add(rs.getString(1).trim());
            }
        } catch (SQLException e) {
            System.err.println("Erreur de récupération des catégories wiki du concept "+concept+".");
        }
        String[] res2 = new String[res.size()];
        res.toArray(res2);
        return res2;
    }

    private Set<LemmaItem> getLemmesDeLaCategorieWiki(String categorieWiki) throws NoBaseException {
        String query = "SELECT l.lemme, l.countTitle, l.countDesc, l.Lex " +
                "FROM ipri_onto_cate_labels c, ipri_onto_cate_lemmes_en l " +
                "WHERE c.idCategorie=l.idCategorie AND c.idRessource=2 AND " +
                "c.CategorieName='"+proc.text.Codecs.escapeHTML(categorieWiki)+"';";
        ResultSet rs = myDB.executeSelection(query);
        Set<LemmaItem> res = new HashSet<LemmaItem>();
        try {
            while (rs.next()) {
                LemmaItem l = new LemmaItem();
                l.setLemmaName(rs.getString(1).trim());
                l.setCountTitle(rs.getInt(2));
                l.setCountDesc(rs.getInt(3));
                l.setLemmaLex(rs.getString(4).trim());
                res.add(l);
            }
        } catch (SQLException e) {
            System.err.println("Erreur de récupération des lemmes de la catégorie "+categorieWiki+".");
        }
        return res;
    }

    public String[] getGeneralisation(String concept) throws NoBaseException {
        CacheDatabase cdb = new CacheDatabase(myDB);
        String[] s = cdb.getCategoriesOnto(concept);
        //System.out.println("!Traitement du concept : " + concept);
        //System.out.println("*" + concept + " : " + s.length + " catégories récupérées dans la base.");
        return s;
    }

    public String[] getSubGeneralisation(String concept) throws NoBaseException {
        CacheDatabase cdb = new CacheDatabase(myDB);
        String[] s = cdb.getCategoriesOnto(concept);
        //System.out.println("##Traitement du concept : " + concept);
        //System.out.println("***" + concept + " : " + s.length + " catégories récupérées dans la base.");
        return s;
    }

    public String[] getGeneralisationStricte(String concept, int niveaux) throws NoBaseException {
        // On va dire que niveaux = 1 pour l'instant.
        String[] premierResultat = getGeneralisation(concept);
        /*System.out.println("*" + premierResultat.length + " catégories trouvées en premier examen.");
        System.out.print("*");
        for (int i = 0; i < premierResultat.length; i++) {
            System.out.print(f(premierResultat[i]) + " ; ");
        }
        System.out.println("");*/
        Set<String> aVerifier = new HashSet<String>();
        for (int i = 0; i < premierResultat.length; i++) {
            aVerifier.add(premierResultat[i]);
        }
        //System.out.println("*" + aVerifier.size() + " catégories uniques.");
        if (aVerifier.size() >= 100) {
            //System.out.println("*Trop de catégories. Mot vide.");
            aVerifier.clear();
        } else {
            Vector<String> liste = new Vector<String>(aVerifier);
            for (int i = 0; i < liste.size(); i++) {
                String s = liste.elementAt(i);
                if (aVerifier.contains(s)) { // sait-on jamais, on l'a peut-être déjà éliminé
                    String[] gene = getSubGeneralisation(s);
                    if (gene.length > 0) {
                        boolean removed = false;
                        for (int j = 0; j < gene.length; j++) {
                            if (aVerifier.contains(gene[j])) {
                                if (!removed) {
                                    //System.out.print("***Nettoyage de la liste des catégories : suppression de ");
                                    removed = true;
                                }
                                aVerifier.remove(gene[j]);
                                //System.out.print(gene[j] + " ; ");
                            }
                        }
                        /*if (removed) {
                            System.out.println("");
                        }*/
                    }
                }
            }
        }
        String[] resultat = new String[aVerifier.size()];
        aVerifier.toArray(resultat);
        //System.out.println("*" + resultat.length + " catégories retenues.");
        //System.out.print("*");
        for (int i = 0; i < resultat.length; i++) {
            //On se passe de l'URI
            resultat[i] = f(resultat[i]);
            //System.out.print(resultat[i] + " ; ");
        }
        //System.out.println("");
        return resultat;
    }

    //pour afficher le nom du concept et non l'url entière
    private String f(String url) {
        String[] s = url.split("\\/");
        return s[s.length - 1];
    }

    public String[] getHigherLevels() throws NoBaseException {
        ArrayList ontologiesIDs = new ArrayList();
        Vector<String> ontologies = new Vector<String>();
        OntologieDatabase odb = new OntologieDatabase(myDB);
        ontologiesIDs = odb.getHigherNodes();
        for (int i = 0; i < ontologiesIDs.size(); i++) {
            int id = Integer.parseInt(ontologiesIDs.get(i).toString());
            ontologies.add(odb.getRessource(id));
        }
        String[] result = new String[ontologies.size()];
        ontologies.toArray(result);
        return result;
    }
}
